function egrad = getPartialEuclideanGradient(problem, x, I, storedb, key)
% Computes the Euclidean gradient of a subset of terms in cost function.
%
% function egrad = getPartialEuclideanGradient(problem, x, I)
% function egrad = getPartialEuclideanGradient(problem, x, I, storedb)
% function egrad = getPartialEuclideanGradient(problem, x, I, storedb, key)
%
% Assume the cost function described in the problem structure is a sum of
% many terms, as
%
%    f(x) = sum_i f_i(x) for i = 1:d,

% where d is specified as d = problem.ncostterms.
% 
% For a subset I of 1:d, getPartialEuclideanGradient obtains the Euclidean
% gradient of the partial cost function
% 
%    f_I(x) = sum_i f_i(x) for i = I.
%
% storedb is a StoreDB object, key is the StoreDB key to point x.
%
% See also: getGradient canGetPartialEuclidean Gradient getPartialGradient

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, June 28, 2016
% Contributors: 
% Change log: 


    % Allow omission of the key, and even of storedb.
    if ~exist('key', 'var')
        if ~exist('storedb', 'var')
            storedb = StoreDB();
        end
        key = storedb.getNewKey();
    end
    
    % Make sure I is a row vector, so that it is natural to loop over it
    % with " for i = I ".
    I = (I(:)).';
    
    
    if isfield(problem, 'partialegrad')
    %% Compute the partial Euclidean gradient using partialegrad.
    
        % Check whether this function wants to deal with storedb or not.
        switch nargin(problem.partialegrad)
            case 2
                egrad = problem.partialegrad(x, I);
            case 3
                % Obtain, pass along, and save the store for x.
                store = storedb.getWithShared(key);
                [egrad, store] = problem.partialegrad(x, I, store);
                storedb.setWithShared(store, key);
            case 4
                % Pass along the whole storedb (by reference), with key.
                egrad = problem.partialegrad(x, I, storedb, key);
            otherwise
                up = MException('manopt:getPartialEuclideanGradient:badpartialegrad', ...
                    'partialegrad should accept 2, 3 or 4 inputs.');
                throw(up);
        end
    
    else
    %% Abandon computing the partial Euclidean gradient.
    
        up = MException('manopt:getPartialEuclideanGradient:fail', ...
            ['The problem description is not explicit enough to ' ...
             'compute the partial Euclidean gradient of the cost.']);
        throw(up);
        
    end
    
end
